#include "Game.h"
#include <sstream>
#include <math.h>

Game::Game(int ScreenWidth, int ScreenHeight, int new_framesPerSecond) {
	framesPerSecond = new_framesPerSecond;
	screen = NULL;

	screenWidth = ScreenWidth;
	screenHeight = ScreenHeight;

	fontRenderer = new FontRenderer();
	fontRenderer->Init();
	utility = new GraphicsUtility();
	samples = new SamplePlayer();//we may pass this in, so we only create it once...
	statusPanel = new StatusPanel(0, (LEVEL_Y_SIZE - Y_OFFSET) * 2, samples);
	currentLevel = new Level(ScreenWidth, ScreenHeight);
	sprites = new SpriteStore();

	gameType = StatusPanel::SinglePlayer;
	csd = sd = NULL;
	//Create all the Lemmings:
	for (int n = 0; n < max_num_lemmings; n++)
		ulemmings.lemmings[n] = new Lemming(6, 9, 0, 0);

	debugMode = false;
	effects_only = false;
	music_now_on=false;
	connected = false;
	ResetMultiplayer();
}

Game::~Game() {
	//Free the surfaces 
	SDL_FreeSurface(screen);

	if (sd != NULL)
		SDLNet_TCP_Close(sd);
	if (csd != NULL)
		SDLNet_TCP_Close(csd);

	SDLNet_Quit();
	delete (fontRenderer);
	delete (statusPanel);
	delete (utility);
	delete (currentLevel);
	delete (samples);

	//delete all the Lemmings:
	for (int n = 0; n < max_num_lemmings; n++)
		delete (ulemmings.lemmings[n]);
}

void Game::Pause() {
	pause = !pause;
}

void Game::Init(StatusPanel::GameType Sender) {
	blueOut = greenOut = releasePoint = numberReleased = totalOut = numHome = 0;
	blueNuked = greenNuked = pause = nukeAll = totalledUp = false;
	lemOutData.new_state = lemInData.new_state = -1;
	lemOutData.index = lemInData.index = 127;
	ticksSinceNukeClick = ticksSinceGameStart = 0;
	music_now_on = false;
	gameType = Sender;
	selectedSprite = NULL;

	if (screen != NULL) {
		SDL_FreeSurface(screen);
		screen = NULL;
	}

	//kill all the Lemmings wuhuhaahaahaa!
	for (int n = 0; n < max_num_lemmings; n++)
		ulemmings.lemmings[n]->SetDead();

	if (screen != NULL)
		SDL_FillRect(screen, &screen->clip_rect, SDL_MapRGB(screen->format, 0x00, 0x00, 0x00));

	minReleaseRate = currentLevel->GetReleaseRate();
	timeRemaining = currentLevel->GetTimeInMinutes() * 60;
	//the following 2 lines ensure we get a lemming straight away
	releaseCounter = (60 - ((int) currentLevel->GetReleaseRate() * 55 / 100));
	fpsCount = framesPerSecond;

	statusPanel->SetGameStats(0, 0, 0, 0, 0, 0);
	statusPanel->viewport.x = currentLevel->GetCamStartX() - screenWidth / 2 + 16;

	if (gameType != StatusPanel::SinglePlayer)
		statusPanel->viewport.x -= screenWidth / 4;

	statusPanel->SetGameMode(gameType);
}

bool Game::CloseConnection() {
	printf("Closing connection\n");
	SDLNet_TCP_Close(sd);
	return true;
}

bool Game::ConnectToGame(StatusPanel::GameType Sender, std::string IPAddress) {
	IPaddress ip;
	connected = false;
	if (!connected) {
		if (gameType == StatusPanel::Server) {
			printf("Server trying to connect\n");
			if (SDLNet_Init() == 0) {
				printf("SDL_NEt has inited\n");
				/* Resolving the host using NULL make network interface to listen */
				if (SDLNet_ResolveHost(&ip, NULL, 2001) == 0) {
					printf("Server Resolved host...");
					/* Open a connection with the IP provided (listen on the host's port) */
					if (!(sd = SDLNet_TCP_Open(&ip))) {
						fprintf(stderr, "SDLNet_TCP_Open: %s\n o", SDLNet_GetError());
						exit(EXIT_FAILURE);
					} else {
						printf("Server now has port open...\n");
						connected = true;
					}
				} else {
					printf("SDLNet_ResolveHost: \n");
				}
			} else {
				printf("SDLNET failed to init\n");
			}
		} else if (gameType == StatusPanel::Client) {
			printf("Client trying to connect\n");
			if (SDLNet_Init() == 0) {
				/* Resolve the host we are connecting to */
				if (SDLNet_ResolveHost(&ip, IPAddress.c_str(), 2001) == 0) {
					fprintf(stderr, "Client Resolved host...");
					/* Open a connection with the IP provided (listen on the host's port) */
					if (!(sd = SDLNet_TCP_Open(&ip))) {
						fprintf(stderr, "SDLNet_TCP_Open: %s\n", SDLNet_GetError());
						connected = false;
					} else {
						fprintf(stderr, "Client connected to server...");
						connected = true;
					}
				}
			} else {
				printf("SDLNET failed to init\n");
			}
			fprintf(stderr, "Client ready to recieve\n");
		}
	}
	return connected;
}

SDL_Surface* Game::GetLevelPreview() {
	return currentLevel->GetLevelPreview();
}

bool Game::WaitForClientToConnect() {
	//now wait until we have a connection
	return (csd = SDLNet_TCP_Accept(sd));
}

bool Game::RequestForData() {
	bool success = false;
	if (gameType == StatusPanel::Server) {
		if (SDLNet_TCP_Recv(csd, &lemInData, data_transfer_size) == data_transfer_size) {
			if (SDLNet_TCP_Send(csd, (void *) &lemOutData, data_transfer_size)
					== data_transfer_size) {
				success = true;
			} else {
				fprintf(stderr, "Server failed to send!\n");
			}
		} else {
			fprintf(stderr, "Server received less than it should!\n");
		}
	} else if (gameType == StatusPanel::Client) {
		if (SDLNet_TCP_Send(sd, (void *) &lemOutData, data_transfer_size) == data_transfer_size) {
			if (SDLNet_TCP_Recv(sd, &lemInData, data_transfer_size) == data_transfer_size) {
				success = true;
			} else {
				fprintf(stderr, "client received not enough\n");
			}
		} else {
			fprintf(stderr, "Client failed to send!\n");
		}
	} else if (gameType == StatusPanel::SinglePlayer)
		success = true;

	return success;
}
/*
bool Game::RequestForData() {
	bool success = false;
	byte dataIn=0;
	byte dataOut=0;

	if (gameType == StatusPanel::Server) {
		if (SDLNet_TCP_Recv(csd, &dataIn, data_transfer_size) == data_transfer_size) {
			if(lemOutData.new_state != 255){
				dataOut = (lemOutData.new_state << 4) | (byte)lemOutData.index;
			}
			if (SDLNet_TCP_Send(csd, (void *) &dataOut, data_transfer_size) == data_transfer_size) {
				success = true;
				lemInData.index = (dataIn & 0xF)+80;
				lemInData.new_state = (dataIn & 0xF0) >> 4;
			} else {
				fprintf(stderr, "Server failed to send!\n");
			}
		} else {
			fprintf(stderr, "Server received less than it should!\n");
		}
	} else if (gameType == StatusPanel::Client) {
		if(lemOutData.new_state != 255){
			dataOut = (lemOutData.new_state << 4) | (byte)(lemOutData.index-80);
		}
		if (SDLNet_TCP_Send(sd, (void *) &dataOut, data_transfer_size) == data_transfer_size) {
			if (SDLNet_TCP_Recv(sd, &dataIn, data_transfer_size) == data_transfer_size) {
				success = true;
				lemInData.index = (dataIn & 0xF);
				lemInData.new_state = (dataIn & 0xF0) >> 4;
			} else {
				fprintf(stderr, "client received not enough\n");
			}
		} else {
			fprintf(stderr, "Client failed to send!\n");
		}
	} else if (gameType == StatusPanel::SinglePlayer)
		success = true;

	return success;
}*/

void Game::LoadLevel(LevelLoader* levelLoader) {
	printf("In game LoadLevel\n");
	printf("loading level: %s\n", levelLoader->GetLevel()->c_str());
	if(currentLevel != NULL){
		printf("currentLevel is Not Null at least\n");
		currentLevel->LoadLevel(levelLoader->GetLevel());
		statusPanel->CreateOverviewImage(currentLevel->GetBackground());
		levelTune = Mix_LoadMUS(levelLoader->GetSong().c_str());
	}
}

void Game::Load2PlayerLevel(LevelLoader* levelLoader) {
	currentLevel->LoadLevel(levelLoader->GetTwoPlayerLevel(0));
	statusPanel->CreateOverviewImage(currentLevel->GetBackground());
	levelTune = Mix_LoadMUS(levelLoader->GetSong().c_str());
}

int Game::getDistanceToLemming(Lemming* l, int xp, int yp){
	int t,v;
	if(xp>l->GetX()){
		t = xp - l->GetX();
		if(yp>l->GetY()){
			v = yp - l->GetY();
		}
		else{
			v = l->GetY() - yp;
		}
	}
	else{
		t = l->GetX()-xp;
		if(yp > l->GetY()){
			v = yp - l->GetY();
		}
		else{
			v = l->GetY() - yp;
		}
	}
	return sqrt((t*t) + (v*v));
}

void Game::SetMouseClick(int x, int y, long leftButtonDown) {
	mouseX = x;
	mouseY = y;
	int iconSelected = -1;
	selectedSprite = NULL;
	int xx = mouseX / 2;
	int yy = mouseY / 2;
	int selectedCount = 0;
	int selectedIndex = 127;

	//correct for (this is shit) multiplayer positioning
	xx -= ((gameType != StatusPanel::SinglePlayer) ? (32 - 4) : 0);//WTF is 32??? it's the width of the totals bar...
	int potdist=999;
	//nothing to do this click, so let the client know:
	lemOutData.new_state = lemOutData.index = 127;
	bool selectThis = false;

	if(leftButtonDown)iconSelected = statusPanel->SelectIcon(x, y);
	iconSelected = statusPanel->GetSelectedIcon();

	//is the mouse over a lemmings?
	for (int n = 0; n < max_num_lemmings; n++) {
		Lemming* potentialNewSelected = ulemmings.lemmings[n];

		//first should we even do this one?
		selectThis = false;

		switch (gameType) {
			case (StatusPanel::Server):
				if (potentialNewSelected->playerNum == 1)
					selectThis = true;
			break;
			case (StatusPanel::Client):
				if (potentialNewSelected->playerNum == 2)
					selectThis = true;
			break;
			case (StatusPanel::SinglePlayer):
				selectThis = true;
			break;
		}

		if (potentialNewSelected != NULL && selectThis && potentialNewSelected->GetAliveState() && !(potentialNewSelected->GetState() & EXPLODING)) {
			if (potentialNewSelected->IsPointWithin(xx + statusPanel->viewport.x, yy + statusPanel->viewport.y)) {
				//this lemming is a candidate
				//so, do we just choose it
				if(selectedSprite == NULL){
					//yes, there's nowt else so a've it
					potdist = getDistanceToLemming(potentialNewSelected,xx+statusPanel->viewport.x,yy+statusPanel->viewport.y);
					selectedSprite = ulemmings.lemmings[n];
					selectedIndex = n;selectedCount++;
				}
				else{
					//this is a bit more difficult then
					//we need to decide if this one is a better
					//candidate or not.

					//the first choice should be which is closer
					int dist = getDistanceToLemming(potentialNewSelected,xx+statusPanel->viewport.x,yy+statusPanel->viewport.y);
					if(dist < potdist){
						//printf("new lem IS closer\n");
						//CLOSER
						//if potential guy is already whatever we're trying to set then don't bother EVER
						if(!(potentialNewSelected->GetState() & iconSelected)){
							//printf("  new lem is not what icon we have\n");
							//he's not one, so use him..perhaps

							//if current selected is NOT blocker and we're NOT set to nuke
							if( !((selectedSprite->GetState()&BLOCKER) && iconSelected==NUKE) ){
								selectedSprite = ulemmings.lemmings[n];
								selectedIndex = n; selectedCount++;
							}
							else if( ((potentialNewSelected->GetState()&BLOCKER) && iconSelected==NUKE) ){
								//if both the same then the closer one wins out!
								selectedSprite = ulemmings.lemmings[n];
								selectedIndex = n; selectedCount++;
							}
							else if(potentialNewSelected->GetState() == selectedSprite->GetState()){
								//if both the same then the closer one wins out!
								selectedSprite = ulemmings.lemmings[n];
								selectedIndex = n; selectedCount++;
							}
							else if(selectedSprite->GetState() & iconSelected){
								//if the one we already have is already this
								selectedSprite = ulemmings.lemmings[n];
								selectedIndex = n; selectedCount++;
							}
						}

					}
					else{
						//NOT CLOSER but still inside, so now further checks:
						//printf("new lem is NOT closer\n");
						//Is the potential one a blocker and are we tyring to nuke? (and it's not already counting down)
						if( (potentialNewSelected->GetState() & BLOCKER) && iconSelected == NUKE && (potentialNewSelected->CountingDown() != -2) ){
							//printf("  but he's a blocker and we're set to nuke\n");
							//yes, so let user nuke blocker
							selectedSprite = ulemmings.lemmings[n];
							selectedIndex = n; selectedCount++;
						}
						else if( !(selectedSprite->GetState() & BLOCKER) && iconSelected == NUKE && (selectedSprite->CountingDown() != -2) ){
							selectedSprite = ulemmings.lemmings[n];
							selectedIndex = n; selectedCount++;
						}
						else if( (selectedSprite->GetState() & BLOCKER) && (iconSelected == BUILDER || iconSelected == MINER || iconSelected == BASHER)){
							selectedSprite = ulemmings.lemmings[n];
							selectedIndex = n; selectedCount++;
						}
						else if(selectedSprite->GetState() & iconSelected){
							//if the one we already have is already this
							selectedSprite = ulemmings.lemmings[n];
							selectedIndex = n; selectedCount++;
						}
					}
				}
			}
		}
	}

	if(ticksSinceNukeClick>0)
		ticksSinceNukeClick++;

	if (leftButtonDown) {
		int doaction = -1;
		switch (statusPanel->SelectIcon(x,y)) {
			case (DIGGER):

				if (selectedSprite != NULL){
					if(currentLevel->ToolAvailable(Level::DIGGER_TOOL)){
						if(!currentLevel->IsPointExpandedMetalArea(selectedSprite->GetX(), selectedSprite->GetY()+2)){
							doaction = 7;
						}
						else
							samples->PlaySample(CHINK);
					}
				}

			break;
			case (MINER):
				if (selectedSprite != NULL){
					if(currentLevel->ToolAvailable(Level::MINER_TOOL)){
						if(!currentLevel->IsPointMetalArea(selectedSprite->GetX() +
										(selectedSprite->GetState() & GOING_LEFT) ? -6 : 8, selectedSprite->GetY()+2)){
							doaction = 6;
						}
						else samples->PlaySample(CHINK);
					}
				}
			break;
			case (BASHER):
				if (selectedSprite != NULL){
					if(currentLevel->ToolAvailable(Level::BASHER_TOOL)){
						if(!currentLevel->IsPointExpandedMetalArea(selectedSprite->GetX() +
											(selectedSprite->GetState() & GOING_LEFT) ? -8 : 10
											, selectedSprite->GetY()-2)){
							doaction = 5;
						}
						else samples->PlaySample(CHINK);
					}
				}
			break;
			case (BUILDER):
				if (selectedSprite != NULL && currentLevel->ToolAvailable(Level::BUILDER_TOOL))
					doaction = 4;
			break;
			case (BLOCKER):
				if (selectedSprite != NULL && currentLevel->ToolAvailable(Level::BLOCKER_TOOL))
					doaction = 3;
			break;
			case (NUKE):
				if (selectedSprite != NULL && currentLevel->ToolAvailable(Level::NUKE_TOOL))
					doaction = 2;
			break;
			case (FLOATER):
				if (selectedSprite != NULL && currentLevel->ToolAvailable(Level::FLOATER_TOOL))
					doaction = 1;
			break;
			case (CLIMBER):
				if (selectedSprite != NULL && currentLevel->ToolAvailable(Level::CLIMBER_TOOL))
					doaction = 0;
			break;

			case (3):
				if(ticksSinceNukeClick > 0 && ticksSinceNukeClick < 8){
					if (gameType == StatusPanel::SinglePlayer) {
						if (!nukeAll) {
							//always play oh no sample if single player
							samples->PlaySample(OHNO);
							nukeAll = true;
							nukeCount = 0;
						}
					} else {
						if (gameType == StatusPanel::Client && !greenNuked) {
							doaction = 8;
							selectedIndex = 127;
						} else if (!blueNuked) {
							doaction = 8;
							selectedIndex = 127;
						}
					}
					ticksSinceNukeClick=0;
				}
				else ticksSinceNukeClick = 1;
			break;

			case (-2):
				if (gameType == StatusPanel::SinglePlayer) {
					pause = !pause;
				}
			break;
		}

		//now, if we need to do something
		if (doaction != 127) {
			//we did something, so save the data
			lemOutData.new_state = (byte) doaction;
			lemOutData.index = (byte) selectedIndex;
		}
	}

	//Shows the type of Lemming in the status panel
	if (selectedSprite != NULL) {
		if ((selectedSprite->GetState() & FALLING))
			statusPanel->SetSelectedStatus("Faller", selectedCount);
		else if ((selectedSprite->GetState() & CLIMBER) && (selectedSprite->GetState() & FLOATER))
			statusPanel->SetSelectedStatus("Athlete", selectedCount);
		else if ((selectedSprite->GetState() & FLOATER))
			statusPanel->SetSelectedStatus("Floater", selectedCount);
		else if ((selectedSprite->GetState() & CLIMBER))
			statusPanel->SetSelectedStatus("Climber", selectedCount);
		else if (selectedSprite->GetState() & WALKING)
			statusPanel->SetSelectedStatus("Walker", selectedCount);
		else if (selectedSprite->GetState() & NUKE)
			statusPanel->SetSelectedStatus("Bomber", selectedCount);
		else if (selectedSprite->GetState() & BUILDER)
			statusPanel->SetSelectedStatus("Builder", selectedCount);
		else if (selectedSprite->GetState() & DIGGER)
			statusPanel->SetSelectedStatus("Digger", selectedCount);
		else if (selectedSprite->GetState() & MINER)
			statusPanel->SetSelectedStatus("Miner", selectedCount);
		else if (selectedSprite->GetState() & BASHER)
			statusPanel->SetSelectedStatus("Basher", selectedCount);
		else if (selectedSprite->GetState() & BLOCKER)
			statusPanel->SetSelectedStatus("Blocker", selectedCount);
		else
			statusPanel->SetSelectedStatus("", 0);
	} else
		statusPanel->SetSelectedStatus("", 0);
}

void Game::SetMousePosition(int x, int y, long leftButtonDown, long rightButtonDown, long lbuttonHeld){

	if (x < ((gameType != StatusPanel::SinglePlayer) ? 64 : 2)) {
		statusPanel->viewport.x += -(rightButtonDown ? 4 : 1);
		if (statusPanel->viewport.x <= 0)
			statusPanel->viewport.x = 0;
	} else if (x > screenWidth * 2 - 2) {
		statusPanel->viewport.x -= -(rightButtonDown ? 4 : 1);
		if (statusPanel->viewport.x > currentLevel->GetBackgroundWidth() - screenWidth)
			statusPanel->viewport.x = currentLevel->GetBackgroundWidth() - screenWidth;
	}

	if (lbuttonHeld && gameType == StatusPanel::SinglePlayer) {
		if (mouseY > statusPanel->ypos) {
			unsigned int rr = currentLevel->GetReleaseRate();
			if (mouseX > 38 && mouseX <= 38 + 20) {
				if (rr < 99)
					currentLevel->SetReleaseRate(rr + 1);
			} else if (mouseX > 8 && mouseX <= 8 + 20) {
				if (rr > minReleaseRate)
					currentLevel->SetReleaseRate(rr - 1);
			}
		}
	}

	if (lbuttonHeld)
		statusPanel->UpdateOverviewOffset(x, y);

	currentLevel->SetViewPort(&statusPanel->viewport);
}

void Game::HandleKey(int key) {
	switch (key) {
	case (SDLK_UP):
		//spriteDebugY += 10;
sprites->TestSprite(1);
		break;

	case (SDLK_DOWN):
		//spriteDebugY -= 10;
sprites->TestSprite(-1);
		break;

	case (SDLK_RIGHT):
		sprites->TestFrame(1);
				//sIconX += 16;
		//if(sIconX > 145)
		//{
		//	sIconX = 32;
		//}
		break;

	case (SDLK_LEFT):
sprites->TestFrame(-1);
				//sIconX -= 15;
		//if(sIconX < 32)
		//{
		//	sIconX = 145 + 16;
		//}
		break;

	case (SDLK_m):
		effects_only = !effects_only;
	break;

	case (SDLK_p):
		Pause();
	break;

	case (SDLK_d):

		//debugMode++;
		//if (debugMode == 4)
		//	debugMode = 0;

	//currentLevel->debug = debugMode>0;
	break;
	}
}

int Game::GetPercentDone() {
	return (numHome * 100) / currentLevel->GetNumLemmings();
}

/*
 * Draw everything on the screen.
 */
void Game::Draw(SDL_Surface *actualScreen) {
	if (screen == NULL) {//this is my hacky way of doubling the pixel size...
		screen = SDL_CreateRGBSurface(actualScreen->flags, 320 - ((gameType
				!= StatusPanel::SinglePlayer) ? 28 : 0), LEVEL_Y_SIZE
				- Y_OFFSET, actualScreen->format->BitsPerPixel,
				actualScreen->format->Rmask, actualScreen->format->Gmask,
				actualScreen->format->Bmask, actualScreen->format->Amask);
	}

	//Clear the screen
	SDL_FillRect(screen, &screen->clip_rect, SDL_MapRGB(screen->format,0x00,0x00,0x33));

	currentLevel->DrawExitsEntrancesWater(screen, NULL);
	utility->apply_surface(0, 0, currentLevel->GetBackground(), screen, &statusPanel->viewport);
	currentLevel->RenderAllOneWayIndicators(screen);

	statusPanel->Draw(actualScreen);

	DrawLemmings(screen, actualScreen);

	if(debugMode == 2)
		currentLevel->DrawDebugStuff(screen);
	if (debugMode == 2)
		sprites->DrawTestAnimationBlocks(screen);
	else if(debugMode == 3)
		sprites->DrawAnimations(screen);

	if (selectedSprite != NULL && debugMode)
		utility->DrawBox(screen, selectedSprite->GetX()
				- statusPanel->viewport.x - 3, selectedSprite->GetY()
				- statusPanel->viewport.y, 10, 12);



	utility->apply_surface_zoomed(
			((gameType != StatusPanel::SinglePlayer) ? (32 * 2) - 4 : 0), 0,
			screen, actualScreen, 2, 0);

	//Draw cursor - cross-hair or box for selected
	if (selectedSprite == NULL) {
		utility->apply_surface(mouseX - 16, mouseY - 16, sprites->cursors, actualScreen,
				&sprites->crossHair.clip);
	} else {
		utility->apply_surface(mouseX - 16, mouseY - 16, sprites->cursors, actualScreen,
				&sprites->LemmingSelected.clip);
		if (debugMode)
			utility->DrawBox(actualScreen, mouseX, mouseY, 10, 16);
	}
}

void Game::DrawLemmings(SDL_Surface* screen, SDL_Surface* actualScreen) {
	Lemming* s;
	int state;
	for (int n = 0; n < max_num_lemmings; n++) {
		s = ulemmings.lemmings[n];
		state = s->GetState();
		if (s->GetAliveState()) {
			//first get the correct clip/sprite
			SpriteStore::AnimObject *animFrame = NULL;
			if (state & WALKING) {
				if (state & GOING_RIGHT) {
					animFrame = &sprites->walkerRight[s->GetFrame()];
				} else {
					animFrame = &sprites->walkerLeft[s->GetFrame()];
				}
			} else if (state & DIGGER) {
				animFrame = &sprites->diggerDowns[s->GetFrame()];
			} else if (state & BUILDER) {
				if (state & GOING_RIGHT) {
					animFrame = &sprites->builderRights[s->GetFrame()];
				} else {
					animFrame = &sprites->builderLefts[s->GetFrame()];
				}
				if (s->GetBuilderRunOut()) {
					samples->PlaySample(TING);
				}
			} else if (state & NOW_CLIMBING) {
				if (state & GOING_RIGHT) {
					animFrame = &sprites->climberRights[s->GetFrame()];
				} else {
					animFrame = &sprites->climberLefts[s->GetFrame()];
				}
			} else if (state & STOPPED_CLIMBING) {
				if (state & GOING_RIGHT) {
					animFrame = &sprites->stoppedClimbingRights[s->GetFrame()];
				} else {
					animFrame = &sprites->stoppedClimbingLefts[s->GetFrame()];
				}
			} else if (state & BLOCKER) {
				animFrame = &sprites->blockers[s->GetFrame()];

				if (debugMode)
					utility->DrawBox(screen, s->GetX()-3 - statusPanel->viewport.x,
										 s->GetY() - statusPanel->viewport.y,
										 6, s->GetHeight()+4);
			} else if (state & FALLING) {
				if ((state & FLOATER) && s->FallingFarEnoughToOpenBrolly()) {
					if (state & GOING_RIGHT) {
						if (state & OPENING_BROLLY) {
							animFrame
							= &sprites->openBrollyRight[s->GetFrame()];
						} else {
							animFrame = &sprites->floaterRight[s->GetFrame()];
						}
					} else {
						if (state & OPENING_BROLLY) {
							animFrame = &sprites->openBrollyLeft[s->GetFrame()];
						} else {
							animFrame = &sprites->floaterLeft[s->GetFrame()];
						}
					}
				} else {
					if (state & GOING_RIGHT) {
						animFrame = &sprites->fallerRight[s->GetFrame()];
					} else {
						animFrame = &sprites->fallerLeft[s->GetFrame()];
					}
				}
			} else if (state & BASHER) {
				if (state & GOING_RIGHT) {
					animFrame = &sprites->basherRights[s->GetFrame()];
				} else {
					animFrame = &sprites->basherLefts[s->GetFrame()];
				}
			} else if (state & MINER) {
				if (state & GOING_RIGHT) {
					animFrame = &sprites->minerRights[s->GetFrame()];
				} else {
					animFrame = &sprites->minerLefts[s->GetFrame()];
				}
			} else if (state & DONE_BUILDING) {
				if (state & GOING_RIGHT) {
					animFrame = &sprites->doneBuildingRights[s->GetFrame()];
				} else {
					animFrame = &sprites->doneBuildingLefts[s->GetFrame()];
				}
			} else if (state & SPLATTED) {
				if (s->GetFrame() == 0)
					samples->PlaySample(SPLAT);
				animFrame = &sprites->splatters[s->GetFrame()];
			} else if ((state & NUKE)) {
				animFrame = &sprites->nukes[s->GetFrame()];
			} else if (state & GOING_HOME) {
				if (state & GOING_RIGHT) {
					animFrame = &sprites->goingHomeRight[s->GetFrame()];
				} else {
					animFrame = &sprites->goingHomeLeft[s->GetFrame()];
				}
			} else if (state & DROWNER) {
				animFrame = &sprites->drowners[s->GetFrame()];
				if (s->GetFrame() == 0)
					samples->PlaySample(SPLASH);
			} else if (state & BURNER) {
				animFrame = &sprites->burners[s->GetFrame()];
				if (s->GetFrame() == 0)
					samples->PlaySample(FIRE);
			}

			//debug point
			if (debugMode)
			utility->DrawBox(screen, s->GetX() - statusPanel->viewport.x,
					s->GetY() + s->GetHeight() - 2 - statusPanel->viewport.y,
												 2, 2);

			if (animFrame != NULL) {
				//finally actually draw the sprite frame:

				if (s->playerNum == 1 || gameType == StatusPanel::SinglePlayer) {
					//draw a blue lemming!
					utility->apply_surface(
							(s->GetX() - statusPanel->viewport.x)
							+ animFrame->offset.x, ((s->GetY()
									- statusPanel->viewport.y))
									+ animFrame->offset.y,
									sprites->player1Sprites, screen, &animFrame->clip);
				} else {
					//draw a green lemming!
					utility->apply_surface_hack((s->GetX()
							- statusPanel->viewport.x) + animFrame->offset.x,
							((s->GetY() - statusPanel->viewport.y))
							+ animFrame->offset.y,
							sprites->player2Sprites, screen, &animFrame->clip);
				}
				statusPanel->DrawMarker(s->GetX(), s->GetY(), actualScreen);
			}

			//if the lemming is moribund (dead or dying, in this case dying)
			short timeToDie = s->CountingDown();
			if ((s->GetRawNukeCount() > 1) && timeToDie != -2 && !(state & EXPLODING)) {
				//draw a countdown timer
				SDL_Rect* timeRect = &sprites->countDown[timeToDie].clip;
				utility->apply_surface(s->GetX() - statusPanel->viewport.x
						+ sprites->countDown[timeToDie].offset.x, s->GetY()
						- statusPanel->viewport.y - 6, sprites->player1Sprites,
						screen, timeRect);
			}

			if (debugMode)
				utility->DrawBox(screen, s->GetX() - statusPanel->viewport.x,
					s->GetY() + s->GetHeight() - 2 - statusPanel->viewport.y,
												 2, 2);


		}

		if (state & EXPLODING) {
			if (s->DrawExplosion(screen, &statusPanel->viewport) == 1) {
				utility->apply_surface(
						s->GetX() - 16 - statusPanel->viewport.x, s->GetY()
						- 10 - statusPanel->viewport.y,
						sprites->player1Sprites, screen, &sprites->bang.clip);
			}
		}
	}
}

/*
 * Adds a new Lemming.
 */
void Game::AddNewLemming() {
	bool green,blue;
	green=blue=false;
	//is this player a blue (player 1) or green (player 2)
	if (gameType != StatusPanel::SinglePlayer) {
		if (greenOut < greenGameMaxLems && blueOut < blueGameMaxLems) {
			if ((numberReleased % 2)) {//green player
				green=true;
			} else {
				blue=true;
			}
		} else {
			if ((numberReleased % 2) && greenGameMaxLems > greenOut) {
				green=true;
			} else if (greenGameMaxLems > blueOut) {
				blue=true;
			}
		}
	}
	else blue=true;

	if(green){
		ulemmings.lemmings[greenOut+80]->SetAlive(currentLevel->GetEntranceX(releasePoint),
												  currentLevel->GetEntranceY(releasePoint) - 16);
		ulemmings.lemmings[greenOut+80]->playerNum = 2;
		ulemmings.lemmings[greenOut+80]->ChangeDirection();
		greenOut++;
	}
	else if(blue){
		ulemmings.lemmings[blueOut]->SetAlive(currentLevel->GetEntranceX(releasePoint),
												 currentLevel->GetEntranceY(releasePoint) - 16);
		ulemmings.lemmings[blueOut]->playerNum = 1;
		blueOut++;
	}

	numberReleased++;
	releasePoint++;
}

int Game::TimeRemaining() {
	return timeRemaining;
}

void Game::SetMusicOn() {
	effects_only = false;
}

void Game::SetMusicOff() {
	effects_only = true;
}

void Game::SetAllSoundOff() {
	//TODO: Implement this function!
}

/*
 * Test to see if the game is complete.
 */
bool Game::IsGameComplete() {
	return (timeRemaining <= 0 || LemmingsSavedPercent() == 100
			||	(numberReleased == currentLevel->GetNumLemmings() && totalOut == 0)
			|| (nukeAll && totalOut == 0)) /*|| WasGameNuked()*/;
}

int Game::LemmingsSavedPercent() {
	return ((numHome * 100) / (currentLevel->GetNumLemmings()));
}

int Game::LemmingsNeededPercent() {
	return (currentLevel->GetLemmingsToBeSaved() * 100) / (currentLevel->GetNumLemmings());
}

bool Game::WasGameNuked() {
	return nukeAll && (nukeCount > 100 && totalOut == 0);
}

/*
 * Returns true if the game was won, false otherwise.
 */
bool Game::WasGameWon() {
	if(gameType == StatusPanel::SinglePlayer)
		return currentLevel->GetLemmingsToBeSaved() <= numHome;
	else {
		if(BlueHome() != GreenHome()){
			if(gameType == StatusPanel::Server)
				return BlueHome() > GreenHome();
			else //if(gameType == StatusPanel::Client)
				return GreenHome() > BlueHome();
		}
		else
		{
			//game draw
			return false;
		}
	}

}

int Game::AnimateLemming(Lemming* s) {
	int gone_home = -1;
	if (s->GetAliveState()) {
		if (!(s->GetState() & EXPLODING))
			totalOut++;

		s->AnimateFrame();
		s->Move(currentLevel->GetBackground());


		//now check the lemming hasn't gone outside of the screen

		if (s->GetX() >= LEVEL_X_SIZE || s->GetX() < 0 || s->GetY()
				>= LEVEL_Y_SIZE-10 || s->GetY() < 0) {
			printf("playing DIE sample\n");
			samples->PlaySample(DIE);
			s->SetDead();
		}
		else{
			for (int m = 0; m < max_num_lemmings; m++) {
				if (ulemmings.lemmings[m] != s) {
					s->CheckForBlocker(ulemmings.lemmings[m]);
				}
			}

			if (nukeAll) {
				if (nukeCount >= 2) {
					//don't nuke them all at the same time,
					//it looks much better if they go off at
					//different times.
					if (!(s->GetState() & GOING_HOME) && !(s->GetState() & NUKE)
							&& !(s->GetState() & EXPLODING) && !(s->GetState()
									& DROWNER) && !(s->GetState() & BURNER)) {
						s->SetToNuke();
						nukeCount = 1;
					}
				}
			}
			//Now check if the lemming has reached the exit
			if (!(s->GetState() & GOING_HOME) && (s->GetState() != DEAD)
					&& !(s->GetState() & EXPLODING)) {
				gone_home = currentLevel->IsPointInHome(s->GetX(), s->GetY());
				if (gone_home != -1) {
					//but a blue entrance or a green one?
					if (gone_home % 2 == 0) {
						gone_home = StatusPanel::Server;
					} else {
						gone_home = StatusPanel::Client;
					}
					s->SetGoingHome();
					samples->PlaySample(BOING);
					numHome++;
				}
			}

			if (currentLevel->IsPointInTrap(s->GetX(), s->GetY())) {
				s->SetDead();
			}
			else if (currentLevel->IsPointInHazard(s->GetX(), s->GetY())
					&& !(s->GetState() & DROWNER) && !(s->GetState() & BURNER)
					&& (s->GetState() != DEAD)) {
				s->SetBurning();
			}
			else if (!(s->GetState() & DROWNER) && (s->GetState() != DEAD)) {
				if(currentLevel->IsPointInWater(s->GetX(), s->GetY()))
					s->SetDrowning();
			}

			//now check for miners,builders,diggers,bashers,bombers
			if (s->GetState() & DIGGER) {
				if (!currentLevel->IsPointMetalArea(s->GetX(), s->GetY())) {
					//where are we digging/affecting backgrund? Lemming or here!?
					if (s->GetFrame() == 8 || s->GetFrame() == 15) {
						SDL_Rect block;
						block.w = 9;
						block.h = 3;
						block.x = s->GetX() - 4;
						block.y = s->GetY() + 7;
						SDL_FillRect(currentLevel->GetBackground(), &block,
								SDL_MapRGB(currentLevel->GetBackground()->format,
										0x00, 0x00, 0x00));
					}
				} else {
					samples->PlaySample(CHINK);
					s->SetToWalker();
				}
			} else if (s->GetState() & MINER) {
				if (s->GetState() & GOING_RIGHT) {
					if (!currentLevel->IsPointMetalArea(s->GetX() + 10, s->GetY())) {
						if (s->Mined()) {
							//draw a black blob in the right place...
							int f = s->GetFrame();
							if (f <= 3)
								utility->apply_surface(s->GetX() + sprites->bashRightMasks[f].offset.x,
													   s->GetY() + sprites->bashRightMasks[f].offset.y,
													   sprites->player1Sprites,
													   currentLevel->GetBackground(),
													   &sprites->bashRightMasks[f].clip,
													   SDL_MapRGB(sprites->player1Sprites->format, 95, 99, 255));
						}
					} else {
						samples->PlaySample(CHINK);
						s->SetToWalker();
					}
				} else if (!currentLevel->IsPointMetalArea(s->GetX() - 8, s->GetY())) {

					if (s->Mined()) {
						//draw a black blob in the right place...
						int f = s->GetFrame();
						if (f <= 3)
						//if (f == 0)
							utility->apply_surface(s->GetX() + sprites->bashLeftMasks[f].offset.x,
												   s->GetY() + sprites->bashLeftMasks[f].offset.y,
												   sprites->player1Sprites,
												   currentLevel->GetBackground(),
												   &sprites->bashLeftMasks[f].clip, SDL_MapRGB(sprites->player1Sprites->format, 95, 99, 255));
					}
				} else {
					samples->PlaySample(CHINK);
					s->SetToWalker();
				}
			} else if ((s->GetState() & BASHER)) {
				SpriteStore::AnimObject* obj = NULL;
				if ((s->GetState() & GOING_RIGHT)) {
					if (!currentLevel->IsPointMetalArea(s->GetX() + 2, s->GetY())
							&& !currentLevel->IsPointOneWayArea(s->GetX() + 2, s->GetY() - 4, 1)) {
						switch (s->GetFrame()) {
						case (3):
						case (19):
						obj = &sprites->bashRightMasks[1];
						break;
						case (4):
						case (20):
						obj = &sprites->bashRightMasks[2];
						break;
						case (5):
						case (21):
						obj = &sprites->bashRightMasks[3];
						break;
						}
						if (obj != NULL)
							utility->apply_surface(s->GetX(),
									s->GetY() + 1,
									sprites->player1Sprites,
									currentLevel->GetBackground(), &obj->clip,
									SDL_MapRGB(sprites->player1Sprites->format, 95,
											99, 255));
					} else {
						samples->PlaySample(CHINK);
						s->SetToWalker();
					}
				} else {
					if (!currentLevel->IsPointMetalArea(s->GetX() - 2, s->GetY())
							&& !currentLevel->IsPointOneWayArea(s->GetX() - 2, s->GetY() - 4, 0)) {
						switch (s->GetFrame()) {
						case (3):
						case (19):
						obj = &sprites->bashLeftMasks[1];
						break;
						case (4):
						case (20):
						obj = &sprites->bashLeftMasks[2];
						break;
						case (5):
						case (21):
						obj = &sprites->bashLeftMasks[3];
						break;
						}
						if (obj != NULL)
							utility->apply_surface(s->GetX() - 11 /*+ obj->offset.x*/,
									s->GetY() + 1,
									sprites->player1Sprites,
									currentLevel->GetBackground(), &obj->clip,
									SDL_MapRGB(sprites->player1Sprites->format, 95,
											99, 255));
					} else {
						samples->PlaySample(CHINK);
						s->SetToWalker();
					}
				}
			}
			else if(s->GetState() & NUKE) {
				if(!nukeAll && s->GetRawNukeCount() == 1)
					samples->PlaySample(OHNO);
			}
			else if (s->GetState() & EXPLODING) {
				if (s->ShallIDrawExplosion()) {
					//TODO: stop this from destroying the metal areas!
					//blow a hole in the landscape
					if(!currentLevel->IsPointExpandedMetalArea(s->GetX() - 8, s->GetY() - 8)){
					utility->apply_surface(s->GetX() - 8, s->GetY() - 8,
							sprites->player1Sprites, currentLevel->GetBackground(),
							&sprites->explosion.clip, SDL_MapRGB(
									sprites->player1Sprites->format, 95, 99, 255));
					}
					//I liked this idea but not true to the amiga (And is crap)
					//currentLevel->RedrawMetalAreas();

					samples->PlaySample(EXPLODE);
				}
			}
		}
	}
	return gone_home;
}

bool Game::DoLemmingAction(LemData* lemData, Lemming* incomingLem, int lemIndex) {
	bool actionDone = false;

	if (lemData->index != 127 && (int) lemData->index == lemIndex) {
		switch (lemData->new_state) {
		case (7):
			//if(!currentLevel->IsPointExpandedMetalArea(incomingLem->GetX(), incomingLem->GetY()+2))
				actionDone = incomingLem->SetToDigger();
			//else
			//	samples->PlaySample(CHINK);
		break;
		case (6):
			//if(!currentLevel->IsPointMetalArea(incomingLem->GetX() +
			//		(incomingLem->GetState() & GOING_LEFT) ? -6 : 8, incomingLem->GetY()+2))
				actionDone = incomingLem->SetToMiner();
			//else
			//	samples->PlaySample(CHINK);
		break;
		case (5):
			//if(!currentLevel->IsPointExpandedMetalArea(incomingLem->GetX() +
			//		(incomingLem->GetState() & GOING_LEFT) ? -8 : 10
			//		, incomingLem->GetY()-2))
				 actionDone = incomingLem->SetToBasher();
			//else
			//	samples->PlaySample(CHINK);
		break;
		case (4):
			actionDone = incomingLem->SetToBuilder();
		break;
		case (3):
			actionDone = incomingLem->SetToBlocker();
		break;
		case (2):
			actionDone = incomingLem->SetToNuke();
		break;
		case (1):
			actionDone = incomingLem->SetToFloater();
		break;
		case (0):
			actionDone = incomingLem->SetToClimber();
		break;
		default:
			printf("!!!!!!!!!!BAD LEMMING STATE SENT!!!!!!!!!!!! index: %d state: %d\n", lemData->index, lemData->new_state);
			break;
		}
	}

	return actionDone;
}

StatusPanel::GameType Game::GetGameType() {
	return gameType;
}

int Game::BlueHome() {
	return statusPanel->BlueHome();
}

int Game::GreenHome() {
	return statusPanel->GreenHome();
}

void Game::ResetMultiplayer() {
	blueGamesWon = greenGamesWon = 0;
	blueGameMaxLems = greenGameMaxLems = min_multiplay_lemmings;
}

/* 
 * Animates the frame or in the case of being a 'client' gets the latest positions
 * from the server
 */
void Game::AnimateFrame() {

	if (!pause) {
		//animate, moving stuff:
		currentLevel->Animate();

		//ensure status panel shows the right stuff:
		statusPanel->SetGameStats(timeRemaining, totalOut, (numHome
							* 100) / currentLevel->GetNumLemmings(),
							(int) currentLevel->GetReleaseRate(),
							currentLevel->Tools(), minReleaseRate);

		//check if we need to start the game proper:
		if (ticksSinceGameStart < 35) {
			if (ticksSinceGameStart == 0) {
				samples->PlaySample(LETSGO);
			}else if (ticksSinceGameStart > 18) {
				currentLevel->AnimateEntrance();
			}else if (ticksSinceGameStart == 18){
				samples->PlaySample(DOOR);
			}
			ticksSinceGameStart++;
		} else {
			//now start the actual game:
			if (!effects_only) {
				if (!music_now_on && Mix_PlayMusic(levelTune, -1) != -1) {
					music_now_on = true;
				}
			} else if (music_now_on) {
				Mix_FadeOutMusic(500);
				music_now_on = false;
			}

			//only count-down if the game is NOT 2 player.l
			if (++fpsCount >= framesPerSecond && (gameType == StatusPanel::SinglePlayer)) {
				timeRemaining--;
				fpsCount = 1;
			}

			if (!nukeAll) {
				if (++releaseCounter >= (60
						- ((int) currentLevel->GetReleaseRate() * 60 / 105))
						&& numberReleased < currentLevel->GetNumLemmings()) {
					if(!debugMode && GetPercentDone()<100)AddNewLemming();
					releaseCounter = 0;
				}

				if (blueNuked && greenNuked) {
					nukeCount = 0;
					samples->PlaySample(OHNO);
					nukeAll = true;
				} else if (lemInData.new_state == 8) {
					if (gameType == StatusPanel::Client) {
						blueNuked = true;
					} else {
						greenNuked = true;
					}
				} else if (lemOutData.new_state == 8) {
					if (gameType == StatusPanel::Client) {
						greenNuked = true;
					} else {
						blueNuked = true;
					}
				}
			} else nukeCount++;

			totalOut = 0;
			for (int n = 0; n < max_num_lemmings; n++) {
				Lemming* s = ulemmings.lemmings[n];

				DoLemmingAction(&lemInData, s, n);

				if (DoLemmingAction(&lemOutData, s, n)) {
					currentLevel->UseTool(
							(Level::ToolType) lemOutData.new_state);
					samples->PlaySample(DOACTION);
				}

				int exit_reached = AnimateLemming(s);
				if (exit_reached != lemming_arrived_home) {
					//lemming's home, save the fact
					if (exit_reached == StatusPanel::Server) {
						//Lem reached a blue home but is he a blue/server?
						if (s->playerNum == 1) {
							statusPanel->AddToBlueTotal(StatusPanel::Server);
						} else {
							statusPanel->AddToBlueTotal(StatusPanel::Client);
						}
					} else {
						//Lem reached a green home but is he a green/server?
						if (s->playerNum == 1) {
							statusPanel->AddToGreenTotal(StatusPanel::Server);
						} else {
							statusPanel->AddToGreenTotal(StatusPanel::Client);
						}
					}
				}
			}

			if (!totalledUp && IsGameComplete()) {
				totalledUp = true;
				if(gameType != StatusPanel::SinglePlayer){
					if (GreenHome() > BlueHome()) {
						greenGamesWon++;
					} else if (GreenHome() < BlueHome()) {
						blueGamesWon++;
					}

					greenGameMaxLems = min_multiplay_lemmings + GreenHome();
					if (greenGameMaxLems > max_num_lemmings / 2)
						greenGameMaxLems = max_num_lemmings / 2;

					blueGameMaxLems = min_multiplay_lemmings + BlueHome();
					if (blueGameMaxLems > max_num_lemmings / 2)
						blueGameMaxLems = max_num_lemmings / 2;
				}
			}
		}
	}
}
