/*
 * Level.cpp
 *
 *  Created on: 24 Jan 2010
 *      Author: mat
 */

#include "Level.h"
#include <sstream>
#include <iostream>

#define Y_OFFSET 8

Level::Level(int ScreenWidth, int ScreenHeight) {
	utility = new GraphicsUtility();
	screenHeight = ScreenHeight;
	screenWidth = ScreenWidth;
	background = NULL;
	levelPreview = NULL;
	viewport = NULL;
	waterFrame = uninteractiveFrame = hazardFrame = 0;
	LoadGraphicalObjects();

	MAX_WATER_FRAME = 7;
	MAX_UNINTERACTIVE_FRAME = 4;
	MAX_HAZARD_FRAME = 4;
	debug = false;
}

Level::~Level() {
	delete(utility);
	SDL_FreeSurface(levelPreview);
	SDL_FreeSurface(background);
}

void Level::SetViewPort(SDL_Rect* rect){
	viewport = rect;
}

int Level::GetEntranceX(int releasePoint){
	return levelStruct->stats.entrance_x[releasePoint % levelStruct->stats.no_entrances];
}

int Level::GetEntranceY(int releasePoint){
	return levelStruct->stats.entrance_y[releasePoint % levelStruct->stats.no_entrances];
}

SDL_Surface* Level::GetBackground(){
	return background;
}

int Level::GetReleaseRate(){
	return levelStruct->stats.release_rate;
}

void Level::SetReleaseRate(int rate){
	levelStruct->stats.release_rate = rate;
}

int Level::GetTimeInMinutes(){
	return levelStruct->stats.time_in_minutes;
}

bool Level::ToolAvailable(ToolType type){
	if(debug==0)
		return tools.at((int)type)>0;
	else
		return true;
}

void Level::ReturnTool(ToolType type){
	tools.at((int)type)++;
}

void Level::UseTool(ToolType type){
	tools.at((int)type)--;
}

std::vector<int>* Level::Tools(){
	return &tools;
}

uint32 Level::GetNumLemmings(){
	return levelStruct->stats.lemmings;
}

uint32 Level::GetLemmingsToBeSaved(){
	return levelStruct->stats.to_be_saved;
}

int Level::GetBackgroundWidth(){
	return background->clip_rect.w;
}

int Level::GetCamStartX(){
	return levelStruct->stats.camera_x;
}

char* Level::GetLevelName(){
	return levelStruct->stats.level_name;
}

char* Level::GetRatingDescription(){
	return levelStruct->stats.rating_description;
}

SDL_Surface* Level::GetLevelPreview(){
	return levelPreview;
}

void Level::AnimateEntrance(){
	if (dooranimframe != 0 && dooranimframe <= 10 && ++dooranimframe == 10)
			dooranimframe = 0;
}

void Level::Animate(){
	if (++oneWayIndicaterFrame > MAX_WATER_FRAME)
		oneWayIndicaterFrame = 0;

	if (++waterFrame > MAX_WATER_FRAME)
		waterFrame = 0;


	AnimFrame* f;
	for(u32 n=0;n<unintFrames.size();n++){
	  f = unintFrames.at(n);
	  f->frame += f->inc;

	  if( f->frame == f->max-1)
		  f->frame=0;

	  //ping pong animation
	  /*if(f->frame == 0){
		f->inc = -f->inc;
	  }
	  else if( f->frame == f->max-1){
		f->inc = -f->inc;
	  }*/
	}

	if (++hazardFrame > (MAX_HAZARD_FRAME + 1))
		hazardFrame = 0;

	//Animate the traps//TODO: should this run backwards when it's finished?
	for (u32 e = 0; e < levelStruct->stats.no_traps; e++) {
		if (trapAnimPoints.at(e) > 0 && ++trapAnimPoints.at(e) == 8)
			trapAnimPoints.at(e) = 0;
	}
}

bool Level::IsPointInWater(int x, int y){
	bool in = false;
	for (u32 w = 0; w < levelStruct->stats.no_waters; w++) {
		if (   x >= levelStruct->stats.water_x1[w]
			&& x < levelStruct->stats.water_x2[w]
			&& y + 8 >= levelStruct->stats.water_y[w] - Y_OFFSET
			&& y + 8 < levelStruct->stats.water_y[w] - Y_OFFSET + 16) {
			in = true;
		}
	}
	return in;
}

bool Level::IsPointExpandedMetalArea(int x, int y) {
	bool in = false;
	y += ((Y_OFFSET * 2) + 3);
	for (u32 nsteel = 0; nsteel < levelStruct->stats.no_steel_areas; nsteel++) {
		if (x - 2 >= levelStruct->stats.steel_area_x1[nsteel]-8 && x - 5
				<= levelStruct->stats.steel_area_x2[nsteel]+8 && y
				>= levelStruct->stats.steel_area_y1[nsteel]-8 && y
				<= levelStruct->stats.steel_area_y2[nsteel]+8) {
			in = true;
			break;
		}
	}
	return in;
}

bool Level::IsPointMetalArea(int x, int y) {
	bool in = false;
	y += ((Y_OFFSET * 2) + 3);
	for (u32 nsteel = 0; nsteel < levelStruct->stats.no_steel_areas; nsteel++) {
		if (x - 2 >= levelStruct->stats.steel_area_x1[nsteel] && x - 5
				<= levelStruct->stats.steel_area_x2[nsteel] && y
				>= levelStruct->stats.steel_area_y1[nsteel] && y
				<= levelStruct->stats.steel_area_y2[nsteel]) {
			in = true;
			break;
		}
	}
	return in;
}

int Level::IsPointInHome(int x, int y) {
	int i = (levelStruct->stats.exit_genus_junction
					& ~LEMMINGS_LEVEL_GENUS_JUNCTIONED_TO_CUSTOM_GENUS_BIT);
	int in = -1;
	y += Y_OFFSET * 2;
	for (u32 e = 0; e < levelStruct->stats.no_exits; e++) {
		// Get the raw number for the exit genus junction
		LEMMINGS_GRAPHICAL_OBJECT_HEADER
				*active_exit_graphical_object =
						(!(levelStruct->stats.exit_genus_junction
								& LEMMINGS_LEVEL_GENUS_JUNCTIONED_TO_CUSTOM_GENUS_BIT))
								? standard_exit_graphical_objects.at(i)
								: custom_exit_graphical_objects.at(i);

		int hx = levelStruct->stats.exit_x[e];
		int hy = levelStruct->stats.exit_y[e];
		if (  x >= (hx + active_exit_graphical_object->active_zone_x1) && x
				<= (hx + active_exit_graphical_object->active_zone_x2) && y
				>= (hy + active_exit_graphical_object->active_zone_y1) && y
				<= (hy + active_exit_graphical_object->active_zone_y2)) {
			in = e;
			break;
		}
	}
	return in;
}

bool Level::IsPointInHazard(int x, int y) {
	bool in = false;
	y += Y_OFFSET * 2;
	for (u32 e = 0; e < levelStruct->stats.no_hazards; e++) {
		int i = (levelStruct->stats.hazard_genus_junctions[levelStruct->stats.hazard_genus[e]]
								& ~LEMMINGS_LEVEL_GENUS_JUNCTIONED_TO_CUSTOM_GENUS_BIT);

		// Return the correct graphical object for the hazard based on the genus junction value.
		LEMMINGS_GRAPHICAL_OBJECT_HEADER
				*active_hazard_graphical_object =
						(!(levelStruct->stats.hazard_genus_junctions[levelStruct->stats.hazard_genus[e]]
								& LEMMINGS_LEVEL_GENUS_JUNCTIONED_TO_CUSTOM_GENUS_BIT))
								? standard_hazard_graphical_objects.at(i)
								: custom_hazard_graphical_objects.at(i);

		int hx = levelStruct->stats.hazard_x[e];
		int hy = levelStruct->stats.hazard_y[e];

		if (  x >= (hx + active_hazard_graphical_object->active_zone_x1) && x
				<= (hx + active_hazard_graphical_object->active_zone_x2) && y
				>= (hy + active_hazard_graphical_object->active_zone_y1) && y
				<= (hy + active_hazard_graphical_object->active_zone_y2)) {
			in = true;
			break;
		}
	}

	return in;
}

bool Level::IsPointInTrap(int x, int y) {
	bool in = false;
	y += Y_OFFSET * 2 + 4;
	for (u32 e = 0; e < levelStruct->stats.no_traps; e++) {
		//only check if trap inactive
		if (trapAnimPoints.at(e) == 0) {
			if (  x >= levelStruct->stats.trap_x[e] && x
					<= levelStruct->stats.trap_x[e] + 5 && y
					>= levelStruct->stats.trap_y[e] && y
					<= levelStruct->stats.trap_y[e] + 5) {
				in = true;
				trapAnimPoints.at(e) = 1;
				break;
			}
		}
	}
	return in;
}

bool Level::IsPointOneWayArea(int x, int y, uint32 direction) {//1 for right, 0 for left
	bool in = false;
	y += Y_OFFSET * 2 + 4;
	for (u32 n = 0; !in && n < levelStruct->stats.no_one_way_areas; n++) {
		if (levelStruct->stats.one_way_area_d[n] != direction) {
			if(  x >= levelStruct->stats.one_way_area_x1[n]-5 && x
				   <= levelStruct->stats.one_way_area_x2[n]+5 && y
				   >= levelStruct->stats.one_way_area_y1[n]-5 && y
				   <= levelStruct->stats.one_way_area_y2[n]+5) {
				in = true;
			}
		}
	}

	return in;
}

void Level::RenderLevelOneWayAreaIndicator(SDL_Surface* screen, int x1, int y1, int x2, int y2, int d, int c) {
	y1 -= Y_OFFSET;
	y2 -= Y_OFFSET;

	s32 dst_x, dst_y; // destination coordinates onto the level.

	s32 start_x = Max(x1, 0);
	s32 start_y = Max(y1, 0);

	s32 bound_x = Min(x2, LEVEL_X_SIZE-1);
	s32 bound_y = Min(y2, LEVEL_Y_SIZE-1);

	u32 ls_x, ls_y;

	u32 draw;
	SDL_Rect rect;
	rect.w = 1;
	rect.h = 1;

	for (dst_y = start_y; dst_y <= bound_y; dst_y++) {
		for (dst_x = start_x; dst_x <= bound_x; dst_x++) {
			ls_x = ((u32) dst_x) & 15;

			if (d == 0) {
				ls_x = 15 - ls_x;
			}

			ls_y = ((u32) dst_y) & 15;

			draw = ((ls_x > 0) && (ls_y >= (ls_x + 8)) && (ls_y <= (16
							- ls_x))) || ((ls_x > 8) && (ls_y >= (ls_x - 8))
							&& (ls_y <= (16 - ls_x)));

			if (draw) {
				//which way to scroll?
				int scroll = oneWayIndicaterFrame+oneWayIndicaterFrame;
				if (d == 0)
					scroll = -scroll;

				if (utility->getpixel(background, dst_x + scroll, dst_y)) {
					if(viewport != NULL){
						rect.x = dst_x - viewport->x + scroll;
						rect.y = dst_y;
					}
					else {
						rect.x = dst_x;
						rect.y = dst_y;
					}

					SDL_FillRect(screen, &rect, SDL_MapRGB(background->format, 255, 255, 0));
				}
			}
		}
	}
}

/****************************
***** ENTRANCES *************
****************************/
void Level::DrawEntrances(SDL_Surface* screen){
	Uint32 e;

	// Get the raw number for the entrance genus junction
	int i = (levelStruct->stats.entrance_genus_junction & ~LEMMINGS_LEVEL_GENUS_JUNCTIONED_TO_CUSTOM_GENUS_BIT);

	// Return the correct graphical object for the entrance based on the genus junction value.
	LEMMINGS_GRAPHICAL_OBJECT_HEADER *active_entrance_graphical_object =
					(!(levelStruct->stats.entrance_genus_junction
							& LEMMINGS_LEVEL_GENUS_JUNCTIONED_TO_CUSTOM_GENUS_BIT))
							? standard_entrance_graphical_objects.at(i)
							: custom_entrance_graphical_objects.at(i);

	// Return the correct graphical object sprite for the entrance based on the genus junction value.
	DSX_SPRITE *active_entrance_graphical_object_sprite =
					(!(levelStruct->stats.entrance_genus_junction
							& LEMMINGS_LEVEL_GENUS_JUNCTIONED_TO_CUSTOM_GENUS_BIT))
							? standard_entrance_graphical_object_sprites.at(i)
							: custom_entrance_graphical_object_sprites.at(i);

	for (e = 0; e < levelStruct->stats.no_entrances; e++) {
		DSX_DrawSpriteFrame(screen, active_entrance_graphical_object,
				active_entrance_graphical_object_sprite, dooranimframe, false,
				levelStruct->stats.entrance_x[e],
				levelStruct->stats.entrance_y[e],
				levelStruct->stats.entrance_palette);
	}
}

/****************************
******* EXITS      **********
*****************************/
void Level::DrawExits(SDL_Surface* screen){
	// Get the raw number for the exit genus junction
	int i = (levelStruct->stats.exit_genus_junction & ~LEMMINGS_LEVEL_GENUS_JUNCTIONED_TO_CUSTOM_GENUS_BIT);

	// Return the correct graphical object for the exit based on the genus junction value.
	LEMMINGS_GRAPHICAL_OBJECT_HEADER *active_exit_graphical_object =
					(!(levelStruct->stats.exit_genus_junction
							& LEMMINGS_LEVEL_GENUS_JUNCTIONED_TO_CUSTOM_GENUS_BIT))
							? standard_exit_graphical_objects.at(i)
							: custom_exit_graphical_objects.at(i);

	// Return the correct graphical object sprite for the exit based on the genus junction value.
	DSX_SPRITE *active_exit_graphical_object_sprite =
					(!(levelStruct->stats.exit_genus_junction
							& LEMMINGS_LEVEL_GENUS_JUNCTIONED_TO_CUSTOM_GENUS_BIT))
							? standard_exit_graphical_object_sprites.at(i)
							: custom_exit_graphical_object_sprites.at(i);

	for (Uint32 e = 0; e < levelStruct->stats.no_exits; e++) {
		DSX_DrawSpriteFrame(screen, active_exit_graphical_object,
				active_exit_graphical_object_sprite,
				active_exit_graphical_object->representing_frame, false,
				levelStruct->stats.exit_x[e],
				levelStruct->stats.exit_y[e],
				levelStruct->stats.exit_palette);
	}
}

/*****************************
 ***** traps  ****************
 *****************************/
void Level::DrawTraps(SDL_Surface* screen){
	for (u32 e = 0; e < levelStruct->stats.no_traps; e++) {
		int i = (levelStruct->stats.trap_genus_junctions[levelStruct->stats.trap_genus[e]]
								& ~LEMMINGS_LEVEL_GENUS_JUNCTIONED_TO_CUSTOM_GENUS_BIT);

		// Return the correct graphical object for the trap based on the genus junction value.
		LEMMINGS_GRAPHICAL_OBJECT_HEADER *active_trap_graphical_object =
						(!(levelStruct->stats.trap_genus_junctions[levelStruct->stats.trap_genus[e]]
								& LEMMINGS_LEVEL_GENUS_JUNCTIONED_TO_CUSTOM_GENUS_BIT))
								? standard_trap_graphical_objects.at(i)
								: custom_trap_graphical_objects.at(i);

		// Return the correct graphical object sprite for the trap based on the genus junction value.
		DSX_SPRITE *active_trap_graphical_object_sprite =
						(!(levelStruct->stats.trap_genus_junctions[levelStruct->stats.trap_genus[e]]
								& LEMMINGS_LEVEL_GENUS_JUNCTIONED_TO_CUSTOM_GENUS_BIT))
								? standard_trap_graphical_object_sprites.at(i)
								: custom_trap_graphical_object_sprites.at(i);



		DSX_DrawSpriteFrame(screen, active_trap_graphical_object,
				active_trap_graphical_object_sprite, trapAnimPoints.at(e),
				false,
				levelStruct->stats.trap_x[e],
				levelStruct->stats.trap_y[e],
				levelStruct->stats.trap_genus_palettes[i]);
	}
}

/****************************
 ***** Hazards **************
 ****************************/
void Level::DrawHazards(SDL_Surface* screen){
	for (u32 e = 0; e < levelStruct->stats.no_hazards; e++) {
		// Get the raw number for the hazard genus junction
		int i = (levelStruct->stats.hazard_genus_junctions[levelStruct->stats.hazard_genus[e]]
								& ~LEMMINGS_LEVEL_GENUS_JUNCTIONED_TO_CUSTOM_GENUS_BIT);

		// Return the correct graphical object for the hazard based on the genus junction value.
		LEMMINGS_GRAPHICAL_OBJECT_HEADER *active_hazard_graphical_object =
						(!(levelStruct->stats.hazard_genus_junctions[levelStruct->stats.hazard_genus[e]]
								& LEMMINGS_LEVEL_GENUS_JUNCTIONED_TO_CUSTOM_GENUS_BIT))
								? standard_hazard_graphical_objects.at(i)
								: custom_hazard_graphical_objects.at(i);

		// Return the correct graphical object sprite for the hazard based on the genus junction value.
		DSX_SPRITE *active_hazard_graphical_object_sprite =
						(!(levelStruct->stats.hazard_genus_junctions[levelStruct->stats.hazard_genus[e]]
								& LEMMINGS_LEVEL_GENUS_JUNCTIONED_TO_CUSTOM_GENUS_BIT))
								? standard_hazard_graphical_object_sprites.at(i)
								: custom_hazard_graphical_object_sprites.at(i);

		DSX_DrawSpriteFrame(screen, active_hazard_graphical_object,
				active_hazard_graphical_object_sprite, hazardFrame, false,
				levelStruct->stats.hazard_x[e],
				levelStruct->stats.hazard_y[e],
				levelStruct->stats.hazard_genus_palettes[i]);
	}
}

void Level::DrawUninteractives(SDL_Surface* screen){
	for (Uint32 e = 0; e < levelStruct->stats.no_uninteractives; e++) {
		// Get the raw number for the uninteractive genus junction
		int i = (levelStruct->stats.uninteractive_genus_junctions[levelStruct->stats.uninteractive_genus[e]]
								& ~LEMMINGS_LEVEL_GENUS_JUNCTIONED_TO_CUSTOM_GENUS_BIT);

		LEMMINGS_GRAPHICAL_OBJECT_HEADER
				*active_uninteractive_graphical_object =
						(!(levelStruct->stats.uninteractive_genus_junctions[levelStruct->stats.uninteractive_genus[e]]
								& LEMMINGS_LEVEL_GENUS_JUNCTIONED_TO_CUSTOM_GENUS_BIT))
								? standard_uninteractive_graphical_objects.at(i)
								: custom_uninteractive_graphical_objects.at(i);

		DSX_SPRITE *active_uninteractive_graphical_object_sprite =
						(!(levelStruct->stats.uninteractive_genus_junctions[levelStruct->stats.uninteractive_genus[e]]
								& LEMMINGS_LEVEL_GENUS_JUNCTIONED_TO_CUSTOM_GENUS_BIT))
								? standard_uninteractive_graphical_object_sprites.at(i)
								: custom_uninteractive_graphical_object_sprites.at(i);
		int fr=0;
		if(e<unintFrames.size())
			fr=unintFrames.at(e)->frame;

		DSX_DrawSpriteFrame(screen, active_uninteractive_graphical_object,
				active_uninteractive_graphical_object_sprite,
				fr, false,
				levelStruct->stats.uninteractive_x[e],
				levelStruct->stats.uninteractive_y[e],
				levelStruct->stats.uninteractive_genus_palettes[i]);
	}
}


/*************************
**** Water ***************
**************************/
void Level::DrawWater(SDL_Surface* screen, uint32 zorder){
	// Get the raw number for the water genus junction
	int i = (levelStruct->stats.water_genus_junction
					& ~LEMMINGS_LEVEL_GENUS_JUNCTIONED_TO_CUSTOM_GENUS_BIT);

	// Return the correct graphical object for the water based on the genus junction value.
	LEMMINGS_GRAPHICAL_OBJECT_HEADER *active_water_graphical_object =
					(!(levelStruct->stats.water_genus_junction
							& LEMMINGS_LEVEL_GENUS_JUNCTIONED_TO_CUSTOM_GENUS_BIT))
							? standard_water_graphical_objects.at(i)
							: custom_water_graphical_objects.at(i);

	// Return the correct graphical object sprite for the water based on the genus junction value.
	DSX_SPRITE *active_water_graphical_object_sprite =
					(!(levelStruct->stats.water_genus_junction
							& LEMMINGS_LEVEL_GENUS_JUNCTIONED_TO_CUSTOM_GENUS_BIT))
							? standard_water_graphical_object_sprites.at(i)
							: custom_water_graphical_object_sprites.at(i);

	for (u32 w = 0; w < levelStruct->stats.no_waters; w++) {
		if (levelStruct->stats.water_z[w] == zorder) {
			DSX_DrawWaterArea(screen,
								active_water_graphical_object,
								active_water_graphical_object_sprite,
								active_water_graphical_object->representing_frame,
								levelStruct->stats.water_x1[w],
								levelStruct->stats.water_x2[w],
								levelStruct->stats.water_y[w]);
		}
	}
}

void Level::DrawExitsEntrancesWater(SDL_Surface* screen, SDL_Surface* b) {

	//if (debug) {



	DrawWater(screen, WATER_Z_BACKGROUND);
	DrawEntrances(screen);
	DrawExits(screen);
	DrawTraps(screen);
	DrawHazards(screen);
	DrawUninteractives(screen);

	if(b != NULL)
		utility->apply_surface(0, 0, b, screen, viewport);

	//RenderAllOneWayIndicators(screen);

	DrawWater(screen, WATER_Z_FOREGROUND);
}



void Level::RenderAllOneWayIndicators(SDL_Surface* screen){
	// Now we need to render the one way areas on top of the level.
	if (levelStruct->one_way_colour != 0) {
		u32 one_way_area;
		for (one_way_area = 0; one_way_area < levelStruct->stats.no_one_way_areas; one_way_area++) {
			RenderLevelOneWayAreaIndicator(screen,
					levelStruct->stats.one_way_area_x1[one_way_area],
					levelStruct->stats.one_way_area_y1[one_way_area],
					levelStruct->stats.one_way_area_x2[one_way_area],
					levelStruct->stats.one_way_area_y2[one_way_area],
					levelStruct->stats.one_way_area_d[one_way_area],
					levelStruct->one_way_colour);
		}
	}
}

void Level::DSX_DrawSpriteFrame(SDL_Surface* screen,
								LEMMINGS_GRAPHICAL_OBJECT_HEADER *graphical_object,
								DSX_SPRITE *sprite_array, int f,
								bool ignore_handles_and_centre,
								int xd, int yd, u16 *palette_to_use) {
	int x = (xd - ((!ignore_handles_and_centre)
							? graphical_object->handle_x
							: (graphical_object->graphic_width / 2)));

	int y = (yd - ((!ignore_handles_and_centre)
							? graphical_object->handle_y
							: (graphical_object->graphic_height / 2)));

	int frame = f;
	if(frame >= graphical_object->no_total_frames-1){
		frame = graphical_object->no_total_frames-1;
	}

	DSX_SPRITE* sprite = &sprite_array[frame];

	u32 c;
	SDL_Rect rect;
	rect.w = rect.h = 1;
	int r, g, b;
	//if (debugMode)
	//	utility->DrawBox(screen, x, y, sprite->width, sprite->height);
	for (int src_x = 0; src_x < sprite->width; src_x++) {
		for (int src_y = 0; src_y < sprite->height; src_y++) {
			int cIndex = sprite->data[src_x + (src_y * (sprite->width))];
			if (cIndex != 0) {
				c = levelStruct->stats.entrance_palette[cIndex];
				//c = palette_to_use[cIndex];
				if(viewport != NULL){
					rect.x = src_x + x - viewport->x;
					rect.y = src_y + y - Y_OFFSET - viewport->y;

					if (rect.x >= 0 && rect.x <= 320) {
						r = (((DSX_COLOUR_RED_PART & c) >> 0) * 255) / 31.0f;
						g = (((DSX_COLOUR_GREEN_PART & c) >> 5) * 255) / 31.0f;
						b = (((DSX_COLOUR_BLUE_PART & c) >> 10) * 255) / 31.0f;
						if (!((r == 0) && (g == 0) && (b == 0)))
							SDL_FillRect(screen, &rect, SDL_MapRGB(screen->format, r, g, b));
					}
				}
				else
				{
					rect.x = src_x + x;
					rect.y = src_y + y - Y_OFFSET;
					r = (((DSX_COLOUR_RED_PART & c) >> 0) * 255) / 31.0f;
					g = (((DSX_COLOUR_GREEN_PART & c) >> 5) * 255) / 31.0f;
					b = (((DSX_COLOUR_BLUE_PART & c) >> 10) * 255) / 31.0f;
					if (!((r == 0) && (g == 0) && (b == 0)))
						SDL_FillRect(screen, &rect, SDL_MapRGB(screen->format, r, g, b));
				}
			}
		}
	}
}

void Level::DSX_DrawSpriteStrip(SDL_Surface* screen, DSX_SPRITE *sprite, int x, int y, int column, const u16* pal) {
	u32 c;
	SDL_Rect rect;
	rect.w = rect.h = 1;

	if (pal == 0)
		pal = sprite->palette;

	int yd;
	int yl = sprite->height;
	int dst_x, dst_y;
	u32 d;

	for (yd = 0; yd < yl; yd++) {
		d = sprite->data[column + (yd * (sprite->width))];

		if (d != 0) {
			dst_y = y + yd;
			dst_x = x;
			c = levelStruct->stats.level_palette[d];
			rect.x = dst_x;
			rect.y = dst_y;
			SDL_FillRect(screen, &rect, SDL_MapRGB(screen->format,
					(((DSX_COLOUR_RED_PART & c) >> 0) * 255) / 31.0f,
					(((DSX_COLOUR_GREEN_PART & c) >> 5) * 255) / 31.0f,
					(((DSX_COLOUR_BLUE_PART & c) >> 10) * 255) / 31.0f));
		}
	}
}

void Level::DSX_DrawWaterArea(SDL_Surface* screen,
		LEMMINGS_GRAPHICAL_OBJECT_HEADER *graphical_object,
		DSX_SPRITE *sprite_array, int frame, int x1, int x2, int y) {
	int xd, yd;
	int water_width = (x2 - x1);

	yd = ((y - graphical_object->handle_y)) - Y_OFFSET;

	int lower_edge = yd - (graphical_object->graphic_height - 1);

	if ((yd > 0) && (lower_edge < screenHeight)) {
		for (int x = 0; x <= water_width; x++) {
			xd = ((x + x1));
			if(viewport != NULL){
				DSX_DrawSpriteStrip(screen, &sprite_array[frame + waterFrame], xd
					- viewport->x, yd - viewport->y, x
					% graphical_object->graphic_width, 0);
			}
			else
			{
				DSX_DrawSpriteStrip(screen, &sprite_array[frame + waterFrame], xd, yd, x
								% graphical_object->graphic_width, 0);
			}
		}
	}


}

long Level::file_size(const char* fileName) {
	long endPos = 0;
	FILE * stream = fopen(fileName, "r");
	if (stream != NULL) {
		fseek(stream, 0L, SEEK_END);
		endPos = ftell(stream);
		fclose(stream);
	}
	return endPos;
}

void Level::DSX_GreedyGatherObjectCategory(std::vector<
		LEMMINGS_GRAPHICAL_OBJECT_HEADER *> *destination_go_vector, // Where will the pointer to this new memory go?
		std::vector<DSX_SPRITE *> *destination_sprite_vector, // Where will the sprites generated go?
		int *quantity_int, // Where's an int to keep track of this stuff?
		const char *source_directory, // Where's the directory?
		const char *source_graphical_object_type, // Whats the filename start?
		u16 *palette_to_set) { // What palette are these sprites going to be using after they're generated?
	// Reset counter
	*quantity_int = 0;

	// Grab all you can
	do {
		// This holds the filename of the file we're going to load
		char incoming_graphical_object_filename[16384];
		sprintf(incoming_graphical_object_filename, "%s%s_%d.lgo",
				source_directory, source_graphical_object_type, *quantity_int);

		if (!file_size(incoming_graphical_object_filename)) {
			return;
		}

		unsigned int graphical_object_filesize;

		// Open the file
		FILE *graphical_object_file = fopen(incoming_graphical_object_filename, "rb");

		// Grab the expected filesize
		fread(&graphical_object_filesize, 4, 1, graphical_object_file);

		rewind(graphical_object_file);

		// Allocate memory for a memory instance of the file
		u8 *graphical_object_memory_instance =
				new u8[graphical_object_filesize];

		// Load the file into memory
		fread(graphical_object_memory_instance, graphical_object_filesize, 1,
				graphical_object_file);

		// Close the file.
		fclose(graphical_object_file);

		// Clonk this pointer onto the graphical object pointer vector.
		destination_go_vector->push_back(
				(LEMMINGS_GRAPHICAL_OBJECT_HEADER *) graphical_object_memory_instance);

		DSX_SPRITE *generated_sprite_array =
						GraphicalObject_ConstructDSSpriteArray(
								(LEMMINGS_GRAPHICAL_OBJECT_HEADER *) graphical_object_memory_instance,
								palette_to_set);

		// Clonk this pointer onto the graphical object pointer vector.
		destination_sprite_vector->push_back(generated_sprite_array);

		(*quantity_int)++;
	} while (1);
}

// This function will gather all of the graphical objects it can from the standard and custom directories.
void Level::LoadGraphicalObjects() {
	const char *standard_loading_location = "data/standard_graphical_objects/";
	const char *custom_loading_location = "data/custom_graphical_objects/";
	const char *object_category_strings[] = {"exit", "entrance", "trap", "hazard", "uninteractive", "water"};

	DSX_GreedyGatherObjectCategory(&standard_exit_graphical_objects,
			&standard_exit_graphical_object_sprites, &no_standard_exits,
			standard_loading_location, object_category_strings[0],
			levelStruct->stats.exit_palette);

	DSX_GreedyGatherObjectCategory(&standard_entrance_graphical_objects,
			&standard_entrance_graphical_object_sprites, &no_standard_entrances,
			standard_loading_location, object_category_strings[1],
			levelStruct->stats.entrance_palette);

	DSX_GreedyGatherObjectCategory(&standard_trap_graphical_objects,
			&standard_trap_graphical_object_sprites, &no_standard_traps,
			standard_loading_location, object_category_strings[2],
			levelStruct->stats.entrance_palette);

	DSX_GreedyGatherObjectCategory(&standard_hazard_graphical_objects,
			&standard_hazard_graphical_object_sprites, &no_standard_hazards,
			standard_loading_location, object_category_strings[3],
			levelStruct->stats.entrance_palette);

	DSX_GreedyGatherObjectCategory(&standard_uninteractive_graphical_objects,
			&standard_uninteractive_graphical_object_sprites, &no_standard_uninteractives,
			standard_loading_location, object_category_strings[4],
			levelStruct->stats.entrance_palette);

	DSX_GreedyGatherObjectCategory(&standard_water_graphical_objects,
			&standard_water_graphical_object_sprites, &no_standard_waters,
			standard_loading_location, object_category_strings[5],
			levelStruct->stats.water_palette);



	DSX_GreedyGatherObjectCategory(&custom_exit_graphical_objects,
			&custom_exit_graphical_object_sprites, &no_custom_exits,
			custom_loading_location, object_category_strings[0],
			levelStruct->stats.exit_palette);

	DSX_GreedyGatherObjectCategory(&custom_entrance_graphical_objects,
			&custom_entrance_graphical_object_sprites, &no_custom_entrances,
			custom_loading_location, object_category_strings[1],
			levelStruct->stats.entrance_palette);

	 DSX_GreedyGatherObjectCategory(&custom_trap_graphical_objects,
			&custom_trap_graphical_object_sprites, &no_custom_traps,
			custom_loading_location,object_category_strings[2],
			levelStruct->stats.entrance_palette);

	 DSX_GreedyGatherObjectCategory(&custom_hazard_graphical_objects,
			&custom_hazard_graphical_object_sprites, &no_custom_hazards,
			custom_loading_location,object_category_strings[3],
			levelStruct->stats.entrance_palette);

	 DSX_GreedyGatherObjectCategory(&custom_uninteractive_graphical_objects,
			&custom_uninteractive_graphical_object_sprites, &no_custom_uninteractives,
			custom_loading_location,object_category_strings[4],
			levelStruct->stats.entrance_palette);

	DSX_GreedyGatherObjectCategory(&custom_water_graphical_objects,
			&custom_water_graphical_object_sprites, &no_custom_waters,
			custom_loading_location, object_category_strings[5],
			levelStruct->stats.entrance_palette);
}

void Level::LoadLevel(std::string* filename) {
	bool valid_load = false;
	dooranimframe = 1;
	oneWayIndicaterFrame=0;

	if (levelStruct != NULL) {
		delete (levelStruct);
		levelStruct = NULL;
	}

	long input_filesize = file_size(filename->c_str());
	if ((input_filesize == 0) || ((u32) input_filesize < sizeof(LEMMINGS_LEVEL_LDS_FILE_V7))) {
		std::cout << "Failed to load level, the filesize was way too small!\n";
	} else
		valid_load = true;

	if (valid_load) {
		if (levelStruct != NULL) delete (levelStruct);
		levelStruct = (LEMMINGS_LEVEL_LDS_FILE_V7 *) new u8[MAIN_MEMORY_CHUNK_SIZE];

		FILE *input_file = fopen(filename->c_str(), "rb");

		// We've got a FILE handle for the input file, let's go to town!
		fread(levelStruct, input_filesize, 1, input_file);
		fclose(input_file);

		enum INVALID_LOAD_REASON {
			INVALID_LOAD_REASON_VALID = 0,
			INVALID_LOAD_REASON_BAD_CHUNK_SIZE,
			INVALID_LOAD_REASON_BAD_VERSION_NUMBER,
			INVALID_LOAD_REASON_CANT_LOAD_ARCHIVE,
			INVALID_LOAD_REASON_ARCHIVE_SIZE_MISMATCH,
			INVALID_LOAD_REASON_ARCHIVE_BAD_VERSION_NUMBER,
			INVALID_LOAD_REASON_COUNT,
		};

		INVALID_LOAD_REASON invalid_load_reason = INVALID_LOAD_REASON_VALID;

		char texture_archive_load_location[1024];
		long texture_archive_input_filesize;

		if ((u32) (input_filesize) != levelStruct->lemmings_level_file_size) {
			valid_load = false;
			std::cout << "Bad Chunk Size\n";
			invalid_load_reason = INVALID_LOAD_REASON_BAD_CHUNK_SIZE;
		} else if (levelStruct->version_number != LEMMINGS_LEVEL_VERSION) {
			valid_load = false;
			invalid_load_reason = INVALID_LOAD_REASON_BAD_VERSION_NUMBER;
		}

		if (valid_load) {
			sprintf(texture_archive_load_location,
					"data/custom_texture_archives/%s.LTA",
					levelStruct->texture_archive_using);
			texture_archive_input_filesize = file_size(
										texture_archive_load_location);

			if ((texture_archive_input_filesize == 0)
					|| ((u32) texture_archive_input_filesize
							< sizeof(LEMMINGS_TEXTURE_ARCHIVE_HEADER))) {
				sprintf(texture_archive_load_location,
						"data/standard_texture_archives/%s.LTA",
						levelStruct->texture_archive_using);

				texture_archive_input_filesize = file_size(
										texture_archive_load_location);
				//valid_load = false;
				if ((texture_archive_input_filesize == 0)
						|| ((u32) texture_archive_input_filesize
								< sizeof(LEMMINGS_TEXTURE_ARCHIVE_HEADER))) {
					valid_load = false;
					invalid_load_reason = INVALID_LOAD_REASON_CANT_LOAD_ARCHIVE;
				}
			}
		}

		LEMMINGS_TEXTURE_ARCHIVE_HEADER *textures = NULL;
		if (valid_load) {
			textures = (LEMMINGS_TEXTURE_ARCHIVE_HEADER *) new u8[texture_archive_input_filesize];

			FILE *texture_archive = fopen(texture_archive_load_location, "rb");

			fread(textures, texture_archive_input_filesize, 1, texture_archive);
			fclose(texture_archive);

			if ((u32)texture_archive_input_filesize != textures->texture_archive_file_size) {
				valid_load = false;
				std::cout << "Archive size mismatch\n";
				invalid_load_reason = INVALID_LOAD_REASON_ARCHIVE_SIZE_MISMATCH;
			}
		}

		if (valid_load) {

			RenderLevel(level_data, levelStruct, textures);
			//Reset trap anim points
			trapAnimPoints.clear();
			for (u32 e = 0; e < levelStruct->stats.no_traps; e++) {
				trapAnimPoints.push_back(0);
			}
			RenderBackgroundAndThumbnail();
		} else {
			printf("failed to load level!\n");
		}

		if(textures != NULL)
			delete (textures);
	}

	tools.clear();

	//load all tools into the easier to use vector:
	for(int n=0; n<8; n++)
		tools.push_back(levelStruct->stats.tool_complement[n]);
	unintFrames.clear();
	for (Uint32 e = 0; e < levelStruct->stats.no_uninteractives; e++) {
		// Get the raw number for the uninteractive genus junction
		int i = (levelStruct->stats.uninteractive_genus_junctions[levelStruct->stats.uninteractive_genus[e]]
								& ~LEMMINGS_LEVEL_GENUS_JUNCTIONED_TO_CUSTOM_GENUS_BIT);

		LEMMINGS_GRAPHICAL_OBJECT_HEADER
		*active_uninteractive_graphical_object =
				(!(levelStruct->stats.uninteractive_genus_junctions[levelStruct->stats.uninteractive_genus[e]]
						& LEMMINGS_LEVEL_GENUS_JUNCTIONED_TO_CUSTOM_GENUS_BIT))
						? standard_uninteractive_graphical_objects.at(i)
						: custom_uninteractive_graphical_objects.at(i);


		AnimFrame* frame =  new AnimFrame();// = new AnimFrame();
		frame->frame = 1;
		frame->inc = 1;
		frame->max = //active_uninteractive_graphical_object->no_secondary_frames
				// active_uninteractive_graphical_object->no_primary_frames;
				 active_uninteractive_graphical_object->no_total_frames;
		printf("adding unin fram e %d tot %d\n",e,frame->max );
		unintFrames.push_back(frame);

	}
}

void Level::CreateThumbnail(){
	SDL_Rect previewRect;
	previewRect.w = previewRect.h = 1;

	int PREVIEW_WIDTH = 640;
	int PREVIEW_HEIGHT = 80;

	int previewdivx = LEVEL_X_SIZE * 10 / PREVIEW_WIDTH * 10;
	int previewdivy = (LEVEL_Y_SIZE-Y_OFFSET) / PREVIEW_HEIGHT;

	if (levelPreview == NULL) {
		levelPreview = SDL_CreateRGBSurface(background->flags, PREVIEW_WIDTH,
				PREVIEW_HEIGHT, background->format->BitsPerPixel,
				background->format->Rmask, background->format->Gmask,
				background->format->Bmask, background->format->Amask);
	}

	//Clear the preview image
	SDL_FillRect(levelPreview, &levelPreview->clip_rect, SDL_MapRGB(levelPreview->format, 0x00, 0x00, 0x33));
	for (int y = 0; y < LEVEL_Y_SIZE-Y_OFFSET; y++) {
	//for (int y = 0; y < background->h; y++) {
		for (int x = 0; x < LEVEL_X_SIZE; x++) {
		//for (int x = 0; x < background->w; x++) {
			previewRect.x = x * 100 / previewdivx;
			previewRect.y = (y) / previewdivy;
			uint32 cd = utility->getpixel(background, x, y );
			if(cd)
				utility->SetPixel(levelPreview, previewRect.x, previewRect.y, cd );
		}
	}
}

void Level::RenderBackgroundAndThumbnail(){
	viewport = NULL;
	waterFrame = uninteractiveFrame = hazardFrame = trapFrame = 0;
	dooranimframe = 1;

	if (background == NULL) {
		background = SDL_CreateRGBSurface(4096, LEVEL_X_SIZE, LEVEL_Y_SIZE, 32, 16711680, 65280, 255, 0);
		background->flags = 4096;
		background->format->colorkey = 0;
	} else {
		SDL_FillRect(background, &background->clip_rect, SDL_MapRGB(background->format, 0x00, 0x00, 0x00));
	}

	DrawExitsEntrancesWater(background, NULL);
	RenderBackgroundDataToSurface(background);
	RenderAllOneWayIndicators(background);
	CreateThumbnail();

	//OUCH! This is a lame waste of CPU effort...
	SDL_FillRect(background, &background->clip_rect, SDL_MapRGB(background->format, 0x00, 0x00, 0x00));
	RenderBackgroundDataToSurface(background);
}

/*
 * This renders the background (the bits that can be destroyed)
 * I.E. not the entrances/exits/etc.
 */
void Level::RenderBackgroundDataToSurface(SDL_Surface* screen){
	SDL_Rect rect;
	rect.w = rect.h = 1;
	for (int y = Y_OFFSET; y < LEVEL_Y_SIZE; y++) {
		for (int x = 0; x < LEVEL_X_SIZE; x++) {
			rect.x = x;
			rect.y = y - Y_OFFSET;//draw offset by 8 pixels, hhmkay?
			u16 c = levelStruct->stats.level_palette[level_data[x][y]];
			if (level_data[x][y]) {
				SDL_FillRect(background, &rect, SDL_MapRGB(
						screen->format, (((DSX_COLOUR_RED_PART & c)
								>> 0) * 255) / 31.0f,
						(((DSX_COLOUR_GREEN_PART & c) >> 5) * 255)
								/ 31.0f, (((DSX_COLOUR_BLUE_PART & c)
								>> 10) * 255) / 31.0f));
			}
		}
	}
}

void Level::RedrawMetalAreas() {
	SDL_Rect rect;
	rect.w = rect.h = 1;
	for (u32 nsteel = 0; nsteel < levelStruct->stats.no_steel_areas; nsteel++) {
		for (int x = levelStruct->stats.steel_area_x1[nsteel]; x < levelStruct->stats.steel_area_x2[nsteel]; x++){
			for (int y = levelStruct->stats.steel_area_y1[nsteel]-2; y
					< levelStruct->stats.steel_area_y2[nsteel]; y++) {
				//now re-draw all those pixels:
				rect.x = x;
				rect.y = y - Y_OFFSET;//draw offset by 8 pixels, hhmkay?
				u16 c = levelStruct->stats.level_palette[level_data[x][y]];
				if (level_data[x][y]) {
					SDL_FillRect(background, &rect, SDL_MapRGB(background->format,
							(((DSX_COLOUR_RED_PART & c) >> 0) * 255) / 31.0f,
							(((DSX_COLOUR_GREEN_PART & c) >> 5) * 255) / 31.0f,
							(((DSX_COLOUR_BLUE_PART & c) >> 10) * 255) / 31.0f));
				}
			}
		}
	}
}

void Level::DrawDebugStuff(SDL_Surface* screen){
	if (true) {
				/*
				for (u32 w = 0; w < levelStruct->stats.no_waters; w++) {
					if (viewport != NULL) {
						utility->DrawBox(screen, levelStruct->stats.water_x1[w]
								- viewport->x, levelStruct->stats.water_y[w]
								- viewport->y - Y_OFFSET,
								levelStruct->stats.water_x2[w]
										- levelStruct->stats.water_x1[w],
								levelStruct->stats.water_y[w] + 16);
					}
				}*/



/*
				SDL_Rect rect;
				rect.w = rect.h = 1;
				for (u32 nsteel = 0; nsteel < levelStruct->stats.no_steel_areas; nsteel++) {
					for (int x = levelStruct->stats.steel_area_x1[nsteel]; x < levelStruct->stats.steel_area_x2[nsteel]; x++){
						for (int y = levelStruct->stats.steel_area_y1[nsteel]-2; y
								< levelStruct->stats.steel_area_y2[nsteel]; y++) {
							//now re-draw all those pixels:
							if (viewport != NULL) {
								rect.x = x - viewport->x;
								rect.y = y - Y_OFFSET - viewport->y;//draw offset by 8 pixels, hhmkay?

								SDL_FillRect(screen, &rect, SDL_MapRGB(background->format,
										111,111,111));

							}
						}
					}
				}*/

				for (u32 s = 0; s < levelStruct->stats.no_steel_areas; s++) {
									if (viewport != NULL) {
										utility->DrawBox(screen,
												levelStruct->stats.steel_area_x1[s] - viewport->x,
												levelStruct->stats.steel_area_y1[s] - viewport->y - Y_OFFSET,
												levelStruct->stats.steel_area_x2[s] - levelStruct->stats.steel_area_x1[s],
												levelStruct->stats.steel_area_y2[s] - levelStruct->stats.steel_area_y1[s] );
									}
								}
			}
}
